
#ifndef square_container_h
#define square_container_h
#include <array>
#include <iostream>
#include <random>


/// Definition of an element of a square
struct SquareElement{
    int row;
    int column;
};

template <size_t hight,size_t width>
/// Container which provides all relevant functions for the assembled squares
class Square{
    
public:
    
    /// Constructor for two element square
    /// @param first_element first square element
    /// @param second_element second square element
    Square(SquareElement first_element,SquareElement second_element):
    size_{2},
    rate_{0.0}{
        //just to make sure
        for (int i=0; i<hight; ++i) {
            for (int j=0; j<width; ++j) {
                placed_elements_[i][j]=false;
            }
        }
        placed_elements_[first_element.row][first_element.column]=true;
        placed_elements_[second_element.row][second_element.column]=true;
    };
    
    
    /// Add an element to the square
    /// @param element element to add
    void AddElements(const SquareElement element){
        placed_elements_[element.row][element.column]=true;
        ++size_;
    }
    
    /// Number of elements placed next an a given position
    /// @param row row of element
    /// @param column column of element
    /// @return number of elements
    const int GetNumberOfNeighbours(const int row,const int column){
        int number=0;
        if (!placed_elements_[row][column]) {
            if (row&&placed_elements_[row-1][column]) {
                ++number;
            }
            if (row!=hight-1&&placed_elements_[row+1][column]) {
                ++number;
            }
            if (column&&placed_elements_[row][column-1]) {
                ++number;
            }
            if (column!=width-1&&placed_elements_[row][column+1]) {
                ++number;
            }
        }
        return number;
    }
    
    /// Calculate the rate at which an event takes place
    /// @param monomers list of possible binding partners
    /// @param critical_size critical monomer size
    /// @param decay_rate decay rate below the critical size
    /// @return rate
    const double GetRate(const std::array<std::array<long,width>,hight>& monomers,const int& critical_size,const double& decay_rate){
        rate_=0;
        if (size_<critical_size) {
            rate_+=decay_rate;
        }
        for (int i=0; i<hight; ++i) {
            for (int j=0;j<width;++j) {
                rate_+=double(GetNumberOfNeighbours(i, j)*monomers[i][j]);
            }
        }
        return rate_;
    }
    
    /// Destroys the square
    /// @param monomers array to which monomers are returned to
    void DestroySquare(std::array<std::array<long,width>,hight>& monomers){
        for (int i=0; i<hight; ++i) {
            for (int j=0;j<width;++j) {
                if (placed_elements_[i][j]) {
                    ++monomers[i][j];
                }
            }
        }
    }
    
    
    /// Performs a random event
    /// @param monomers array with all active monomers
    /// @param critical_size critical square size
    /// @param decay_rate decay rate
    /// @param engine random engine reference
    /// @param finished_squares number of finished squares
    /// @param square_was_not_finished return if square was finished
    /// @param monomer return binding monomer
    ///@return if monomer can be removed
    const bool EventTakePlace(std::array<std::array<long,width>,hight>& monomers,const int& critical_size,const double& decay_rate,std::mt19937& engine,long& finished_squares,bool& square_was_not_finished,SquareElement& monomer){
        double random_number=std::uniform_real_distribution<double>(0,rate_)(engine);
        double sum=0;
        if (size_<critical_size) {
            if (random_number<decay_rate) {
                DestroySquare(monomers);
                return true;
            }
            sum+=decay_rate;
        }
        if(++size_==width*hight){
            ++finished_squares;
            square_was_not_finished=false;
            for (int i=0; i<hight; ++i) {
                for (int j=0;j<width;++j) {
                    if (!placed_elements_[i][j]) {
                        monomer.row=i;
                        monomer.column=j;
                    }
                }
            }
            return true;
        }
        for (int i=0; i<hight; ++i) {
            for (int j=0;j<width;++j) {
                    sum+=double(GetNumberOfNeighbours(i, j)*monomers[i][j]);
                    if (random_number<sum) {
                        monomer.row=i;
                        monomer.column=j;
                        placed_elements_[i][j]=true;
                        return false;
                    }
            }
        }
        
        std::cout<<"Error has occured! Program will be aborted!\n";
        exit(EXIT_FAILURE);
        
    }

    /// Access to elements via [][] notation
    std::array<bool,hight>& operator[](const int index)const{
        return placed_elements_[index];
    }
    /// Access to elements via [] notation per element
    bool operator[](SquareElement element){
        return placed_elements_[element.row][element.column];
    }
    /// assigment operator
    void operator=(Square other){
        this->placed_elements_=other.placed_elements_;
        this->size_=other.size_;
        this->rate_=other.rate_;
    }
private:
    //private members viariables for a square
    std::array<std::array<bool,width>,hight> placed_elements_;
    int size_;
    double rate_;
};

#endif /* square_container_h */
